//
//  GMSChatViewController.m
//  GMS Sample
//
/*
 Version: 1.0
 
 Disclaimer: IMPORTANT:  This software is supplied to you by Genesys
 Telecommunications Laboratories Inc ("Genesys") in consideration of your agreement
 to the following terms, and your use, installation, modification or redistribution
 of this Genesys software constitutes acceptance of these terms.  If you do not
 agree with these terms, please do not use, install, modify or redistribute this
 Genesys software.
 
 In consideration of your agreement to abide by the following terms, and subject
 to these terms, Genesys grants you a personal, non-exclusive license, under
 Genesys's copyrights in this original Genesys software (the "Genesys Software"), to
 use, reproduce, modify and redistribute the Genesys Software, with or without
 modifications, in source and/or binary forms; provided that if you redistribute
 the Genesys Software in its entirety and without modifications, you must retain
 this notice and the following text and disclaimers in all such redistributions
 of the Genesys Software.
 
 Neither the name, trademarks, service marks or logos of Genesys Inc. may be used
 to endorse or promote products derived from the Genesys Software without specific
 prior written permission from Genesys.  Except as expressly stated in this notice,
 no other rights or licenses, express or implied, are granted by Genesys herein,
 including but not limited to any patent rights that may be infringed by your
 derivative works or by other works in which the Genesys Software may be
 incorporated.
 
 The Genesys Software is provided by Genesys on an "AS IS" basis.  GENESYS MAKES NO
 WARRANTIES, EXPRESS OR IMPLIED, INCLUDING WITHOUT LIMITATION THE IMPLIED
 WARRANTIES OF NON-INFRINGEMENT, MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 PURPOSE, REGARDING THE GENESYS SOFTWARE OR ITS USE AND OPERATION ALONE OR IN
 COMBINATION WITH YOUR PRODUCTS.
 
 IN NO EVENT SHALL GENESYS BE LIABLE FOR ANY SPECIAL, INDIRECT, INCIDENTAL OR
 CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
 GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 ARISING IN ANY WAY OUT OF THE USE, REPRODUCTION, MODIFICATION AND/OR
 DISTRIBUTION OF THE GENESYS SOFTWARE, HOWEVER CAUSED AND WHETHER UNDER THEORY OF
 CONTRACT, TORT (INCLUDING NEGLIGENCE), STRICT LIABILITY OR OTHERWISE, EVEN IF
 GENESYS HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 
 Copyright (C) 2013 Genesys Inc. All Rights Reserved.
 */

#import "GMSChatViewController.h"
#import "DDCometMessage.h"
#import "UIBubbleTableView.h"
#import "GMSLogViewController.h"
#import "GMSRequestViewController.h"
#import "GMSAppDelegate.h"


#define refreshURL      @"/refresh"
#define startTypingURL  @"/startTyping"
#define stopTypingURL   @"/stopTyping"
#define disconnectURL   @"/disconnect"

@interface GMSChatViewController ()

@property (nonatomic, unsafe_unretained) BOOL chatActive;

@end

@implementation GMSChatViewController
// instance variables declared in implementation context
{
    NSString *startSessionURL;
    NSString *cometURL;
    NSString *userHeader;
    NSDictionary *chatParams;
    
    NSMutableArray *localBubbleData;            //displayed in UIBubbleTableView, synchronized with remoteBubbleData
    NSMutableArray *remoteBubbleData;           //received by CometD
    DDCometClient *cometClient;
    NSString *cometClientID;
    NSString *gmsSession;
    NSString *agentName;
}


- (void)startWithURL:(NSString *)startURL cometURL:(NSString *)cURL userHeader:(NSString *)uHeader chatParams:(NSDictionary *)cParams
{    
    startSessionURL = startURL;
    cometURL = cURL;
    userHeader = uHeader;
    chatParams = cParams;
    
    //Init CometD
    if (cometClient == nil)
	{
 		cometClient = [[DDCometClient alloc] initWithURL:[NSURL URLWithString:cometURL] gmsHeader:userHeader];
		cometClient.delegate = self;
		[cometClient scheduleInRunLoop:[NSRunLoop currentRunLoop] forMode:NSDefaultRunLoopMode];
        [cometClient handshake];
	}    
}

#pragma mark Getters and Setters

- (void) setChatActive:(BOOL)chatActive
{
    if (chatActive) {
        _doneButton.enabled = YES;
        _textField.enabled = YES;
    }
    else {
        _doneButton.enabled = NO;
        _textField.enabled = NO;
    }
}


#pragma mark - View life cycle

- (void)viewDidLoad
{
    [super viewDidLoad];
    
    if (localBubbleData == nil) {
        localBubbleData = [[NSMutableArray alloc] init];
    }

    if (remoteBubbleData == nil) {
        remoteBubbleData = [[NSMutableArray alloc] init];
    }

    _tableView.bubbleDataSource = self;
    
    // The line below sets the snap interval in seconds. This defines how the bubbles will be grouped in time.
    // Interval of 120 means that if the next messages comes in 2 minutes since the last message, it will be added into the same group.
    // Groups are delimited with header which contains date and time for the first message in the group.
    
    _tableView.snapInterval = 120;
    
    // The line below disables avatar support. Avatar can be specified for each bubble with .avatar property of NSBubbleData.
    // Avatars are enabled for the whole table at once. If particular NSBubbleData misses the avatar, a default placeholder will be set (missingAvatar.png)
    
    _tableView.showAvatars = NO;
    
    // Uncomment the line below to add "Now typing" bubble
    // Possible values are
    //    - NSBubbleTypingTypeSomebody - shows "now typing" bubble on the left
    //    - NSBubbleTypingTypeMe - shows "now typing" bubble on the right
    //    - NSBubbleTypingTypeNobody - no "now typing" bubble
    
    _tableView.typingBubble = NSBubbleTypingTypeNobody;
    
    // Keyboard events
    
    [[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(keyboardWasShown:) name:UIKeyboardWillShowNotification object:nil];
    [[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(keyboardWillBeHidden:) name:UIKeyboardWillHideNotification object:nil];

    self.chatActive = NO;
    
    // Move button lower in case of iPhone 5 (4-inch display)
    CGRect screenBounds = [[UIScreen mainScreen] bounds];
    if (screenBounds.size.height == 568) {
        // code for 4-inch screen
        CGRect frame = [_textField frame];
        frame.origin.y += 88;
        [_textField setFrame:frame];
    }

}

- (void)didReceiveMemoryWarning
{
    [super didReceiveMemoryWarning];
    // Dispose of any resources that can be recreated.
}

- (void)viewDidUnload {
    [[NSNotificationCenter defaultCenter] removeObserver:self];

    _tableView.bubbleDataSource = nil;
    localBubbleData = nil;
    remoteBubbleData = nil;
    [self setTableView:nil];
    [self setTextField:nil];
    [self setStatusLabel:nil];
    [self setDoneButton:nil];
    [super viewDidUnload];
}

#pragma mark - UIBubbleTableViewDataSource implementation

- (NSInteger)rowsForBubbleTable:(UIBubbleTableView *)tableView
{
    return [localBubbleData count];
}

- (NSBubbleData *)bubbleTableView:(UIBubbleTableView *)tableView dataForRow:(NSInteger)row
{
    return [localBubbleData objectAtIndex:row];
}

#pragma mark - DDComet Client Delegate methods

- (void)submit2GMS:(NSString *)urlStr body:(NSDictionary *)bodyDict
{
    //Find GMSRequestViewController
    GMSAppDelegate *appDelegate = (GMSAppDelegate*)[[UIApplication sharedApplication] delegate];
    UITabBarController *tb = (UITabBarController *)appDelegate.window.rootViewController;
    for (id tbc in tb.viewControllers) {
        if ([tbc isKindOfClass:[UINavigationController class]] && [[tbc viewControllers][0] isKindOfClass:[GMSRequestViewController class]]) {
            GMSRequestViewController *gmsR = [tbc viewControllers][0];
            [gmsR.dMenu submitRequest2GMS:urlStr method:@"POST" params:bodyDict headers:@{@"gms_user" : userHeader}];
        }
    }
}

- (void) logRequest:(NSString *)txt2Log direction:(int)dir
{
    //Update Log View
    [[NSNotificationCenter defaultCenter] postNotificationName:kUpdateNotification
                                                        object:self
                                                      userInfo:@{@"direction":@(dir), @"text":txt2Log}];
}

- (void)showError:(NSString *)errorStr
{
    UIAlertView *alert = [[UIAlertView alloc]
                          initWithTitle: @"GMS Chat Server Error"
                          message: errorStr
                          delegate: nil
                          cancelButtonTitle:@"OK"
                          otherButtonTitles:nil];
    dispatch_async(dispatch_get_main_queue(), ^{
        [alert show];
    });    
}

- (void)cometClientHandshakeDidSucceed:(DDCometClient *)client
{
	[self logRequest:@"Handshake succeeded, waiting for an agent..." direction:fromGMS];
    _statusLabel.text = @"Waiting for an agent...";
    
    self.chatActive = YES;
    
    cometClientID = client.clientID;
	
	[client subscribeToChannel:@"/_genesys" target:self selector:@selector(chatMessageReceived:)];
    
    //Start Chat
    NSMutableDictionary *chatDict = [NSMutableDictionary dictionaryWithDictionary:chatParams];
    [chatDict setValue:@"true" forKey:@"_verbose"];
    [chatDict setValue:@"comet" forKey:@"notify_by"];
    [self submit2GMS:startSessionURL body:chatDict];
}

- (void)cometClient:(DDCometClient *)client handshakeDidFailWithError:(NSError *)error
{
	[self logRequest:@"CometD Handshake failed" direction:fromGMS];
}

- (void)cometClientConnectDidSucceed:(DDCometClient *)client
{
	[self logRequest:@"CometD Connect succeeded" direction:fromGMS];
}

- (void)cometClient:(DDCometClient *)client connectDidFailWithError:(NSError *)error
{
	[self logRequest:[NSString stringWithFormat:@"CometD Connect failed, error: %@", error] direction:fromGMS];
}

- (void)cometClient:(DDCometClient *)client subscriptionDidSucceed:(DDCometSubscription *)subscription
{
	[self logRequest:@"CometD Subsrciption succeeded" direction:fromGMS];
}

- (void)cometClient:(DDCometClient *)client subscription:(DDCometSubscription *)subscription didFailWithError:(NSError *)error
{
	[self logRequest:[NSString stringWithFormat:@"CometD Subsrciption failed, error: %@", error] direction:fromGMS];
}

- (void)chatMessageReceived:(DDCometMessage *)message
{    
	if (message.successful == nil) {
        NSDictionary *jsonResponse = [message.data objectForKey:@"message"];
        
        if (jsonResponse != nil) {
            NSArray *transcript = (NSArray *) [jsonResponse objectForKey:@"transcriptToShow"];
            
            if (transcript != nil) {
                for (int i=0; i < transcript.count; i++) {
                    NSArray *row = (NSArray *) [transcript objectAtIndex:i];
                    for (int j=0; j < row.count; j++) {
                        NSString *object = [row objectAtIndex:j];
                        if([object isEqualToString:@"Notice.Joined"]) {
                            agentName = [row objectAtIndex:j + 1];                            
                            _statusLabel.text = [NSString stringWithFormat:@"%@ %@", [row objectAtIndex:j+1], [row objectAtIndex:j+2]];
                            break;
                        }
                        if([object isEqualToString:@"Notice.TypingStarted"]) {
                            //React only to messages from agent
                            if ([[row objectAtIndex:j + 1] isEqualToString:agentName]) {
                                _tableView.typingBubble = NSBubbleTypingTypeSomebody;
                                [_tableView reloadData];
                                [self scrollToBottom];
                            }
                             break;
                        }
                        if([object isEqualToString:@"Notice.TypingStopped"]) {
                            _tableView.typingBubble = NSBubbleTypingTypeNobody;
                            [_tableView reloadData];
                            break;
                        }
                        if([object isEqualToString:@"Message.Text"]) {
                            if (agentName != nil) { //Ignore messages before the agent joins in
                                if ([agentName isEqualToString:(NSString *) [row objectAtIndex:j + 1]]) { //Do not display messages apart from the agent's
                                    // Display new message from AGENT
                                    NSBubbleData *rcvBubble = [NSBubbleData dataWithText:[row objectAtIndex:j + 2]
                                                                                    date:[NSDate dateWithTimeIntervalSinceNow:0]
                                                                                    type:BubbleTypeSomeoneElse];
                                    [localBubbleData addObject:rcvBubble];
                                    [remoteBubbleData addObject:rcvBubble];
                                    _tableView.typingBubble = NSBubbleTypingTypeNobody;
                                    [_tableView reloadData];
                                    [self scrollToBottom];
                                 }
                                if ([(NSString *) [row objectAtIndex:j + 4] isEqualToString:@"CLIENT"]) {
                                    // Display new message from CLIENT
                                    NSBubbleData *rcvBubble = [NSBubbleData dataWithText:[row objectAtIndex:j + 2]
                                                                                    date:[NSDate dateWithTimeIntervalSinceNow:0]
                                                                                    type:BubbleTypeMine];
                                    [remoteBubbleData addObject:rcvBubble];
                                    if (![localBubbleData isEqual:remoteBubbleData]) {
                                        //Sync
                                        [self logRequest:@"Synchronized remote chat messages order..." direction:fromGMS];
                                        localBubbleData = [NSMutableArray arrayWithArray:remoteBubbleData];
                                        [_tableView reloadData];
                                        [self scrollToBottom];
                                    }
                                }
                            }
                            break;
                        }
                        if([object isEqualToString:@"Notice.Left"]) {
                            if (agentName != nil) { //Ignore messages before the agent joins in
                                if ([agentName isEqualToString:(NSString *) [row objectAtIndex:j + 1]]) { //Do not display messages apart from the agent's
                                    _statusLabel.text = [NSString stringWithFormat:@"%@ %@", [row objectAtIndex:j+1], [row objectAtIndex:j+2]];
                                    [self performSelector:@selector(endChat:) withObject:nil afterDelay:5.0];
                                }
                            }
                            break;
                        }
                    }
                }
            }
        }
        
    }
	else if (![message.successful boolValue])
		_statusLabel.text = @"Unable to send message";
}

//- (void)membershipMessageReceived:(DDCometMessage *)message
//{
//}

-(void) scrollToBottom
{
    int lastSection=[_tableView numberOfSections]-1;
    int lastRowNumber = [_tableView numberOfRowsInSection:lastSection]-1;
    NSIndexPath* ip = [NSIndexPath indexPathForRow:lastRowNumber inSection:lastSection];
    [_tableView scrollToRowAtIndexPath:ip atScrollPosition:UITableViewScrollPositionTop animated:YES];
}

#pragma mark - Keyboard events

- (void)keyboardWasShown:(NSNotification*)aNotification
{
    NSDictionary* info = [aNotification userInfo];
    CGSize kbSize = [[info objectForKey:UIKeyboardFrameBeginUserInfoKey] CGRectValue].size;
    
    [UIView animateWithDuration:0.2f animations:^{
        
        CGRect frame = _textField.frame;
        frame.origin.y -= kbSize.height - [[[super tabBarController] tabBar] frame].size.height;
        _textField.frame = frame;
        
        frame = _tableView.frame;
        frame.size.height -= kbSize.height - [[[super tabBarController] tabBar] frame].size.height;
        _tableView.frame = frame;
    }];
}

- (void)keyboardWillBeHidden:(NSNotification*)aNotification
{
    NSDictionary* info = [aNotification userInfo];
    CGSize kbSize = [[info objectForKey:UIKeyboardFrameBeginUserInfoKey] CGRectValue].size;
    
    [UIView animateWithDuration:0.2f animations:^{
        
        CGRect frame = _textField.frame;
        frame.origin.y += kbSize.height - [[[super tabBarController] tabBar] frame].size.height;
        _textField.frame = frame;
        
        frame = _tableView.frame;
        frame.size.height += kbSize.height - [[[super tabBarController] tabBar] frame].size.height;
        _tableView.frame = frame;
    }];
}



#pragma mark - TextField delegates

- (void)textFieldDidBeginEditing:(UITextField *)textField
{
    [self submit2GMS:[startSessionURL stringByAppendingString:startTypingURL] body:@{@"_verbose": @"true"}];
}

- (void)textFieldDidEndEditing:(UITextField *)textField
{
    [self submit2GMS:[startSessionURL stringByAppendingString:stopTypingURL] body:@{@"_verbose": @"true"}];
}

//Dismiss keyboard when Return key is pressed
- (BOOL)textFieldShouldReturn:(UITextField *)textField {
    
    _tableView.typingBubble = NSBubbleTypingTypeNobody;
    
    if ([_textField.text length] > 0) {
        
        [self submit2GMS:[startSessionURL stringByAppendingString:refreshURL] body:@{@"_verbose": @"true", @"message" : _textField.text}];
        
        NSBubbleData *sayBubble = [NSBubbleData dataWithText:_textField.text date:[NSDate dateWithTimeIntervalSinceNow:0] type:BubbleTypeMine];
        [localBubbleData addObject:sayBubble];
        [_tableView reloadData];
        [self scrollToBottom];
    }
    
    _textField.text = @"";
    [_textField resignFirstResponder];
    
    return YES;
}


#pragma mark - Action Clear

- (IBAction)clearChat:(id)sender {
    [localBubbleData removeAllObjects];
    [remoteBubbleData removeAllObjects];
    [_tableView reloadData];
}

- (IBAction)endChat:(id)sender {
    _statusLabel.text = @"Chat Ended";
    self.chatActive = NO;
    [self submit2GMS:[startSessionURL stringByAppendingString:disconnectURL] body:@{@"_verbose": @"true"}];
    [cometClient unsubsubscribeFromChannel:@"/_genesys" target:self selector:@selector(chatMessageReceived:)];
    [cometClient disconnect];
    cometClient.delegate = nil;
    cometClient = nil;
}


@end
